<?php
/**
 * batch_upload.php — Secure multi-image uploader using 'batch' as reference.
 * Saves .jpg images and updates d_jobs.bd_photo with a single URL.
 */

if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_FILES['images'])) {
    header('Content-Type: application/json');
    header('Access-Control-Allow-Origin: *');

    // ===== CONFIG =====
    $db_host = 'localhost';
    $db_name = 'rhedlkwp_cav';
    $db_user = 'rhedlkwp_Rhedrick';
    $db_pass = 'Avr2025!Cav2026!';
    $baseUrl  = 'https://verify.rhedrickinterglobal.com/cav/imagery/';
    $targetDir = __DIR__ . '/cav/imagery/';
    if (!is_dir($targetDir)) mkdir($targetDir, 0777, true);

    try {
        $db = new PDO("mysql:host=$db_host;dbname=$db_name;charset=utf8mb4", $db_user, $db_pass);
        $db->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
    } catch (Exception $e) {
        echo json_encode([['error' => 'DB connection failed: ' . $e->getMessage()]]);
        exit;
    }

    $signature = $_POST['signature'] ?? '';
    $allowed = ['Royusuf', 'Rhedrick', 'Renewal', 'TierraCasa', 'OBrien', 'XGlobal'];
    if (!in_array($signature, $allowed)) {
        echo json_encode([['error' => 'Invalid signature']]);
        exit;
    }

    $results = [];

    foreach ($_FILES['images']['tmp_name'] as $i => $tmpPath) {
        $originalName = $_FILES['images']['name'][$i];
        $batch = preg_replace('/[^A-Za-z0-9_\-]/', '', pathinfo($originalName, PATHINFO_FILENAME));
        $newFilename = $batch . '.jpg';
        $uploadPath = $targetDir . $newFilename;
        $fileUrl = $baseUrl . $newFilename;

        if (!$batch) {
            $results[] = ['file' => $originalName, 'status' => '❌ invalid name'];
            continue;
        }

        // Verify record
        $stmt = $db->prepare("SELECT signature FROM d_jobs WHERE batch = ?");
        $stmt->execute([$batch]);
        $job = $stmt->fetch(PDO::FETCH_ASSOC);

        if (!$job) {
            $results[] = ['file' => $originalName, 'status' => '❌ batch not found'];
            continue;
        }

        if (trim($job['signature']) !== $signature) {
            $results[] = ['file' => $originalName, 'status' => '❌ signature mismatch'];
            continue;
        }

        // Save image
        if (!move_uploaded_file($tmpPath, $uploadPath)) {
            $results[] = ['file' => $originalName, 'status' => '❌ upload failed'];
            continue;
        }

        // Update bd_photo with single URL
        $stmt = $db->prepare("UPDATE d_jobs SET bd_photo = ? WHERE batch = ?");
        $stmt->execute([$fileUrl, $batch]);

        $results[] = ['file' => $originalName, 'status' => '✅ uploaded'];
    }

    echo json_encode($results, JSON_PRETTY_PRINT);
    exit;
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
<meta charset="UTF-8">
<title>Batch Image Upload</title>
<style>
body {
  font-family: "Inter", system-ui, sans-serif;
  background: #f8fafc;
  color: #333;
  display: flex;
  flex-direction: column;
  align-items: center;
  padding: 40px;
}
.container {
  background: white;
  border-radius: 20px;
  box-shadow: 0 8px 25px rgba(0,0,0,0.08);
  padding: 30px;
  max-width: 520px;
  width: 100%;
  transition: transform .2s ease;
}
.container:hover { transform: translateY(-3px); }
h2 {
  text-align: center;
  margin-bottom: 1em;
  color: #444;
}
#dropzone {
  border: 2px dashed #6c63ff;
  border-radius: 15px;
  padding: 40px;
  text-align: center;
  cursor: pointer;
  transition: background 0.25s, border-color 0.25s;
}
#dropzone.dragover {
  background: #eef0ff;
  border-color: #4b47ff;
}
#preview {
  display: grid;
  grid-template-columns: repeat(auto-fill, minmax(90px, 1fr));
  gap: 10px;
  margin-top: 15px;
}
.preview-item {
  background: #fafafa;
  border: 1px solid #e5e7eb;
  border-radius: 12px;
  padding: 5px;
  font-size: 12px;
  text-align: center;
  color: #555;
}
.preview-item img {
  width: 100%;
  height: 80px;
  object-fit: cover;
  border-radius: 8px;
}
select, button {
  width: 100%;
  padding: 10px;
  border: 1px solid #ccc;
  border-radius: 10px;
  margin-top: 15px;
  font-size: 15px;
}
button {
  background: #6c63ff;
  color: white;
  border: none;
  cursor: pointer;
  transition: background 0.3s ease;
}
button:hover {
  background: #5149e0;
}
.progress-wrapper {
  background: #e4e7ec;
  border-radius: 10px;
  height: 10px;
  margin-top: 20px;
  overflow: hidden;
}
.progress-bar {
  height: 100%;
  background: linear-gradient(90deg, #6c63ff, #a599ff);
  width: 0%;
  transition: width 0.3s ease;
}
#progressText {
  text-align: center;
  font-size: 13px;
  margin-top: 8px;
  white-space: pre-line;
  line-height: 1.4;
}
</style>
</head>
<body>

<div class="container">
  <h2>Batch Image Upload</h2>
  <div id="dropzone">Drop images here or click to browse</div>
  <input type="file" id="fileInput" multiple accept="image/*" hidden>
  <select id="signature">
    <option value="">Select Signature</option>
    <option>Rhedrick</option>
  </select>
  <div id="preview"></div>
  <button id="uploadBtn">Upload Files</button>
  <div class="progress-wrapper"><div class="progress-bar" id="progressBar"></div></div>
  <div id="progressText"></div>
</div>

<script>
const dropzone = document.getElementById('dropzone');
const fileInput = document.getElementById('fileInput');
const preview = document.getElementById('preview');
const uploadBtn = document.getElementById('uploadBtn');
const progressBar = document.getElementById('progressBar');
const progressText = document.getElementById('progressText');
let selectedFiles = [];

// ==== Compress Image ====
function compressImage(file, maxKB = 300, maxDim = 1600) {
  return new Promise(resolve => {
    const reader = new FileReader();
    reader.onload = e => {
      const img = new Image();
      img.onload = () => {
        let w = img.width, h = img.height;
        if (w > maxDim || h > maxDim) {
          const scale = Math.min(maxDim / w, maxDim / h);
          w = Math.round(w * scale);
          h = Math.round(h * scale);
        }
        const canvas = document.createElement('canvas');
        canvas.width = w; canvas.height = h;
        const ctx = canvas.getContext('2d');
        ctx.drawImage(img, 0, 0, w, h);

        let quality = 0.9;
        (function compressLoop() {
          canvas.toBlob(blob => {
            if (!blob) return resolve(file);
            const sizeKB = blob.size / 1024;
            if (sizeKB <= maxKB || quality <= 0.4) {
              resolve(new File([blob], file.name.replace(/\.[^/.]+$/, '') + '.jpg', { type: 'image/jpeg' }));
            } else {
              quality -= 0.05;
              compressLoop();
            }
          }, 'image/jpeg', quality);
        })();
      };
      img.src = e.target.result;
    };
    reader.readAsDataURL(file);
  });
}

// ==== Drag & Drop ====
dropzone.onclick = () => fileInput.click();
['dragover', 'dragleave', 'drop'].forEach(ev => dropzone.addEventListener(ev, e => e.preventDefault()));
dropzone.addEventListener('dragover', () => dropzone.classList.add('dragover'));
dropzone.addEventListener('dragleave', () => dropzone.classList.remove('dragover'));
dropzone.addEventListener('drop', e => {
  dropzone.classList.remove('dragover');
  handleFiles(e.dataTransfer.files);
});
fileInput.onchange = e => handleFiles(e.target.files);

function handleFiles(files) {
  selectedFiles = Array.from(files);
  preview.innerHTML = selectedFiles.map(f => `
    <div class="preview-item">
      <img src="${URL.createObjectURL(f)}">
      <div>${f.name}<br><small>${(f.size/1024).toFixed(1)} KB</small></div>
    </div>`).join('');
}

// ==== Upload ====
uploadBtn.onclick = async () => {
  const signature = document.getElementById('signature').value;
  if (!signature) return alert('Please select a signature.');
  if (!selectedFiles.length) return alert('No images selected.');

  progressText.textContent = "Compressing images...";
  const compressed = [];
  for (const f of selectedFiles) compressed.push(await compressImage(f));

  preview.innerHTML = compressed.map(f => `
    <div class="preview-item">
      <img src="${URL.createObjectURL(f)}">
      <div>${f.name}<br><small>${(f.size/1024).toFixed(1)} KB</small></div>
    </div>`).join('');

  const formData = new FormData();
  compressed.forEach(f => formData.append('images[]', f));
  formData.append('signature', signature);

  progressText.textContent = "Uploading...";
  const xhr = new XMLHttpRequest();
  xhr.open('POST', window.location.href);
  xhr.upload.addEventListener('progress', e => {
    if (e.lengthComputable) {
      const percent = (e.loaded / e.total) * 100;
      progressBar.style.width = percent + "%";
      progressText.textContent = `Uploading... ${percent.toFixed(1)}%`;
    }
  });

  xhr.onload = () => {
    if (xhr.status === 200) {
      try {
        const result = JSON.parse(xhr.responseText);
        progressText.textContent = result.map(r => `${r.file}: ${r.status}`).join('\n');
      } catch {
        progressText.textContent = xhr.responseText;
      }
      progressBar.style.width = "100%";
    } else {
      progressText.textContent = "❌ Upload failed.";
    }
  };
  xhr.send(formData);
};
</script>
</body>
</html>