<?php
/**
 * csv_to_d_jobs_uploader.php
 * Single-file PHP uploader that imports a CSV into table d_jobs and updates/inserts into jobs.
 * - Missing CSV columns become NULL.
 * - Empty CSV values become NULL.
 * - st_photo, bd_photo, coord are ALWAYS NULL.
 * - Allows overrides (signature, tat, status, client) from the upload form.
 *
 * NOTE: For production move DB creds out of code and secure this endpoint.
 */

session_start();

// ---------------------------
// DB CONFIG - change these
// ---------------------------
require_once __DIR__ . '/con.php';

$dbHost = constant('hostc');
$dbName = constant('db_namec');
$dbUser = constant('db_userc');
$dbPass = constant('db_passc');

$dsn = "mysql:host={$dbHost};dbname={$dbName};charset=utf8mb4";

// Expected d_jobs columns (order doesn't matter) --------------------------------
$expectedCols = [
    'trxnref','customer_phone','customer_title','customer_name','first_name','last_name','customer_address','address_status','residential','customer_resides','nearest_bstp','landmark','alias','company_name','city','state','ver_stat_desc','building_condition','building_finishing','building_completion','building_type2','customer_rel_with_address','occupation','owner_relationship','area_profile','interviewed','interviewee','interviewee_surname','interviewee_other_names','interviewee_relationship','interviewee_address','interviewee_telephone','visitation_date','visitation_time','staff_met','staff_met_comment','address_visited','building_type','colour_of_building','sure_of_address','reason','st_photo','bd_photo','coord','comments','agent','l_agent','client','post_date','post_by','status','signature','tat','batch'
];

// Jobs table columns for fallback insertion --------------------------------------
$jobsCols = [
    'trxnref','customer_name','first_name','last_name','state','city','phone_no','occupation','address','landmark','alias','company','agent','l_agent','client','post_by','signature','post_date','status','comment','reverif','batch','saved_coord'
];

// helpers ------------------------------------------------------------------------
function normalizeHeader($h){
    return trim(strtolower(str_replace(["\r","\n","\t"],'',$h)));
}

function tryParseDate($value){
    // return Y-m-d or null
    if ($value === null || $value === '') return null;
    $ts = strtotime($value);
    if ($ts === false) return null;
    return date('Y-m-d', $ts);
}
function tryParseTime($value){
    // return h:ia or null
    if ($value === null || $value === '') return null;
    $ts = strtotime($value);
    if ($ts === false) return null;
    return date('h:ia', $ts);
}

function formatPostDateFromParts($datePart, $timePart){
    // both already standardized as strings
    $d = tryParseDate($datePart);
    $t = tryParseTime($timePart);
    if ($d && $t) {
        // produce "YYYY-MM-DD HH:MMam/pm"
        $dt = DateTime::createFromFormat('Y-m-d h:ia', "$d $t");
        if (!$dt) return null;
        return $dt->format('Y-m-d h:ia');
    }
    return null;
}

function randomTimeBetween($startHour=9, $endHour=18){
    $today = new DateTime('today');
    $start = (clone $today)->setTime($startHour,0);
    $end = (clone $today)->setTime($endHour,0);
    $now = new DateTime();
    // if now is before start, return start
    if ($now < $start) return $start->format('h:ia');
    // adjust end to min(now, end)
    $effectiveEnd = $now < $end ? $now : $end;
    if ($effectiveEnd < $start) return $start->format('h:ia');
    $startTs = $start->getTimestamp();
    $endTs = $effectiveEnd->getTimestamp();
    $randTs = rand($startTs, $endTs);
    return (new DateTime())->setTimestamp($randTs)->format('h:ia');
}

function ensureUtf8($value) {
    if ($value === null || $value === '') {
        return $value;
    }

    // already valid UTF-8
    if (mb_check_encoding($value, 'UTF-8')) {
        return $value;
    }

    // convert common Excel encodings → UTF-8
    return mb_convert_encoding($value, 'UTF-8', 'Windows-1252, ISO-8859-1, ISO-8859-15');
}


// ---------------------------
// Handle upload POST
// ---------------------------
$messages = [];
$results = [];

if ($_SERVER['REQUEST_METHOD'] === 'POST'){
    // read overrides from form (allow blank = do not override)
    $overrideSignature = isset($_POST['override_signature']) && $_POST['override_signature'] !== '' ? $_POST['override_signature'] : null;
    $overrideTAT       = isset($_POST['override_tat']) && $_POST['override_tat'] !== '' ? $_POST['override_tat'] : null;
    $overrideStatus    = isset($_POST['override_status']) && $_POST['override_status'] !== '' ? $_POST['override_status'] : null;
    $overrideClient    = isset($_POST['override_client']) && $_POST['override_client'] !== '' ? $_POST['override_client'] : null;

    if (!isset($_FILES['csvfile']) || $_FILES['csvfile']['error'] !== UPLOAD_ERR_OK) {
        $messages[] = ['type'=>'error','text'=>'Please choose a valid CSV file.'];
    } else {
        $tmp = $_FILES['csvfile']['tmp_name'];
        $filename = $_FILES['csvfile']['name'];
        // open CSV
        if (($handle = fopen($tmp, 'r')) === false) {
            $messages[] = ['type'=>'error','text'=>'Could not open uploaded CSV.'];
        } else {
            // read header
            $header = fgetcsv($handle);
            if ($header === false) {
                $messages[] = ['type'=>'error','text'=>'CSV appears empty.'];
                fclose($handle);
            } else {
                // normalize header
                $originalHeader = $header;
                $headerMap = [];
                foreach ($header as $i=>$h) {
                    $normalized = normalizeHeader($h);
                    $headerMap[$normalized] = $i;
                }

                // create PDO
                try {
                    $pdo = new PDO($dsn, $dbUser, $dbPass, [
                        PDO::ATTR_ERRMODE            => PDO::ERRMODE_EXCEPTION,
                        PDO::ATTR_EMULATE_PREPARES   => false,  // avoids truncation edge cases
                        PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
                    ]);
                    $pdo->exec("SET NAMES utf8mb4 COLLATE utf8mb4_unicode_ci");
                } catch (Exception $e) {
                    $messages[] = ['type'=>'error','text'=>'DB connection failed: '.$e->getMessage()];
                    fclose($handle);
                    goto render;
                }

                // start processing rows
                $rowIndex = 0;

                // prepare d_jobs insert - build dynamic placeholder list
                $placeholders = implode(',', array_fill(0, count($expectedCols), '?'));
                $insertSql = 'INSERT INTO d_jobs ('.implode(',',$expectedCols).') VALUES ('.$placeholders.')';
                $insertStmt = $pdo->prepare($insertSql);

                // We'll need queries against jobs
                $selectJob = $pdo->prepare('SELECT * FROM jobs WHERE trxnref = ? LIMIT 1');
                $updateJob = $pdo->prepare('UPDATE jobs SET status = ? WHERE trxnref = ?');
                // build insert for jobs fallback with columns that exist in jobsCols
                $jobsInsertSql = 'INSERT INTO jobs ('.implode(',',$jobsCols).') VALUES ('.implode(',', array_fill(0, count($jobsCols), '?')).')';
                $insertJobStmt = $pdo->prepare($jobsInsertSql);

                // begin transaction
                $pdo->beginTransaction();
                try {
                    while (($row = fgetcsv($handle)) !== false) {
                        $rowIndex++;
                        // build data for d_jobs following expectedCols
                        $drow = [];
                        // create an associative map for original columns (normalized keys)
                        $assoc = [];
                        foreach ($headerMap as $norm=>$idx) {
                            $value = isset($row[$idx]) ? $row[$idx] : null;
                            $assoc[$norm] = ensureUtf8($value);
                        }

                        // For each expected column, prefer CSV value if present; else NULL
                        foreach ($expectedCols as $col) {
                            $lc = strtolower($col);

                            $val = null; // default NULL in strict mode

                            // Try exact lowercase key
                            if (array_key_exists($lc, $assoc)) {
                                $v = $assoc[$lc];
                                // treat empty string as NULL
                                if ($v === '') {
                                    $val = null;
                                } else {
                                    $val = ensureUtf8($v);
                                }
                            } else {
                                // Try variants: without underscores
                                $alt = str_replace('_','',$lc);
                                if (array_key_exists($alt, $assoc)) {
                                    $v = $assoc[$alt];
                                    if ($v === '') $val = null;
                                    else $val = $v;
                                } else {
                                    // column not present in CSV -> NULL (strict mode)
                                    $val = null;
                                }
                            }

                            // Special formatting for dates/times if present
                            if ($lc === 'visitation_date' && $val !== null) {
                                $parsed = tryParseDate($val);
                                if ($parsed) $val = $parsed;
                            }
                            if ($lc === 'visitation_time' && $val !== null) {
                                $parsed = tryParseTime($val);
                                if ($parsed) $val = $parsed;
                            }

                            $drow[$col] = $val;
                        }

                        // Ensure st_photo, bd_photo, coord are always NULL
                        $drow['st_photo'] = null;
                        $drow['bd_photo'] = null;
                        $drow['coord'] = null;

                        // post_date logic: if both visitation_date and visitation_time present, use them
                        $postDate = null;
                        $vd = isset($drow['visitation_date']) ? $drow['visitation_date'] : null;
                        $vt = isset($drow['visitation_time']) ? $drow['visitation_time'] : null;
                        if ($vd !== null && $vt !== null){
                            $pd = formatPostDateFromParts($vd, $vt);
                            if ($pd) $postDate = $pd;
                        }
                        if (!$postDate) {
                            // pick today's date and random time between 9am and now (or 6pm cap)
                            $today = new DateTime('today');
                            $dateStr = $today->format('Y-m-d');
                            $randTime = randomTimeBetween(9,18); // returns h:ia
                            $postDate = "$dateStr $randTime";
                            // also set visitation_date/time to these if they were NULL
                            if ($vd === null) $drow['visitation_date'] = $dateStr;
                            if ($vt === null) $drow['visitation_time'] = $randTime;
                        }

                        $drow['post_date'] = $postDate;

                        // If there is an existing jobs row, get agent and post_by from that row to insert into d_jobs
                        $tr = isset($drow['trxnref']) && $drow['trxnref'] !== null ? $drow['trxnref'] : null;
                        $agentVal = null;
                        $postByVal = null;
                        if ($tr) {
                            $selectJob->execute([$tr]);
                            $jobRow = $selectJob->fetch(PDO::FETCH_ASSOC);
                            if ($jobRow) {
                                // update jobs status to Done
                                try {
                                    $updateJob->execute(['Done', $tr]);
                                } catch (Exception $e) {
                                    // non-fatal: log and continue
                                }
                                $agentVal = isset($jobRow['agent']) && $jobRow['agent'] !== '' ? $jobRow['agent'] : null;
                                $postByVal = isset($jobRow['post_by']) && $jobRow['post_by'] !== '' ? $jobRow['post_by'] : null;
                            } else {
                                // insert fallback row into jobs using CSV data
                                // build jobs insert values in the expected order
                                $ji = [];
                                foreach ($jobsCols as $jc) {
                                    $lower = strtolower($jc);
                                    $value = null;

                                    // Map from CSV assoc - try a few common keys
                                    if (array_key_exists($lower, $assoc) && $assoc[$lower] !== '') $value = $assoc[$lower];
                                    else if (array_key_exists(str_replace('_','',$lower), $assoc) && $assoc[str_replace('_','',$lower)] !== '') $value = $assoc[str_replace('_','',$lower)];
                                    else {
                                        switch ($lower){
                                            case 'phone_no':
                                                if (isset($assoc['customer_phone']) && $assoc['customer_phone'] !== '') $value = $assoc['customer_phone'];
                                                break;
                                            case 'address':
                                                if (isset($assoc['customer_address']) && $assoc['customer_address'] !== '') $value = $assoc['customer_address'];
                                                break;
                                            case 'company':
                                                if (isset($assoc['company_name']) && $assoc['company_name'] !== '') $value = $assoc['company_name'];
                                                break;
                                            case 'saved_coord':
                                                if (isset($assoc['coord']) && $assoc['coord'] !== '') $value = $assoc['coord'];
                                                else $value = null;
                                                break;
                                            case 'post_by':
                                                // Always use signature (override or CSV signature)
                                                if ($overrideSignature !== null) {
                                                    $value = $overrideSignature;
                                                } else {
                                                    $value = isset($assoc['signature']) && $assoc['signature'] !== '' 
                                                                ? $assoc['signature'] 
                                                                : null;
                                                }
                                                break;
                                        }
                                    }

                                    if ($jc === 'post_date') {
                                        // Use computed post date
                                        $ji[] = $postDate;
                                    } elseif ($jc === 'status') {
                                        $ji[] = 'Done';
                                    } elseif ($jc === 'trxnref') {
                                        $ji[] = $tr ?: null;
                                    } elseif ($jc === 'landmark') {
                                        // Keep earlier behavior: force landmark to 'N/A'
                                        $ji[] = 'N/A';
                                    } else {
                                        // ensure nulls are preserved
                                        $ji[] = $value;
                                    }
                                }
                                try {
                                    $insertJobStmt->execute($ji);
                                } catch (Exception $e) {
                                    // don't break the whole import - report on this row
                                    $results[] = ['row'=>$rowIndex,'trxnref'=>$tr,'status'=>'job_insert_failed','error'=>$e->getMessage()];
                                }
                            }
                        }

                        // override d_jobs agent and post_by if we got them from jobs table and CSV didn't provide them
                        // CSV post_by should take precedence if present
                        if ((isset($drow['post_by']) && $drow['post_by'] !== null)) {
                            // keep CSV post_by
                        } elseif (!empty($postByVal)) {
                            $drow['post_by'] = $postByVal;
                        } else {
                            $drow['post_by'] = null;
                        }

                        if ((isset($drow['agent']) && $drow['agent'] !== null)) {
                            // keep CSV agent
                        } elseif (!empty($agentVal)) {
                            $drow['agent'] = $agentVal;
                        } else {
                            $drow['agent'] = null;
                        }

                        // --------------------
                        // APPLY OVERRIDES (Strict: overrides win even if CSV had a value)
                        // --------------------
                        if ($overrideSignature !== null) {
                            $drow['signature'] = $overrideSignature;
                        }

                        if ($overrideTAT !== null) {
                            $drow['tat'] = $overrideTAT;
                        }

                        if ($overrideStatus !== null) {
                            $drow['status'] = $overrideStatus;
                        }

                        if ($overrideClient !== null) {
                            $drow['client'] = $overrideClient;
                        }

                        // Ensure landmark remains as CSV value if present; but earlier code forced 'N/A' in jobs fallback only.
                        // d_jobs.landmark will use CSV value if present, else NULL (strict mode)
                        // No change required here.

                        // Build insert values in order of expectedCols
                        $insertValues = [];
                        foreach ($expectedCols as $col) {
                            $v = array_key_exists($col, $drow) ? $drow[$col] : null;
                            // For safety: st_photo, bd_photo, coord must be NULL
                            if (in_array($col, ['st_photo','bd_photo','coord'])) $v = null;

                            // Normalize empty-string to NULL (if any slipped through)
                            if ($v === '') $v = null;

                            $insertValues[] = $v;
                        }

                        // execute insert
                        try {
                            foreach ($insertValues as $i => $v) {
                                if (is_null($v)) {
                                    $insertStmt->bindValue($i + 1, null, PDO::PARAM_NULL);
                                } else {
                                    $insertStmt->bindValue($i + 1, $v);
                                }
                            }
                            $insertStmt->execute();

                            $results[] = ['row'=>$rowIndex,'trxnref'=>$tr,'status'=>'Inserted'];
                        } catch (Exception $e) {
                            $results[] = ['row'=>$rowIndex,'trxnref'=>$tr,'status'=>'d_jobs_insert_failed','Error'=>$e->getMessage()];
                        }
                    } // end while rows

                    $pdo->commit();
                } catch (Exception $e) {
                    $pdo->rollBack();
                    $messages[] = ['type'=>'error','text'=>'Transaction failed: '.$e->getMessage()];
                }

                fclose($handle);
                // set success messages and results variable for UI
                $messages[] = ['type'=>'success','text'=>'CSV processed. Rows Read: '.$rowIndex];
                $_SESSION['import_results'] = $results ?? [];
            }
        }
    }
}

render:
?>
<!doctype html>
<html>
<head>
    <meta charset="utf-8">
    <title>CSV → d_jobs Uploader</title>
    <meta name="viewport" content="width=device-width,initial-scale=1">
    <style>
        body{font-family:system-ui, -apple-system, 'Segoe UI', Roboto, Arial; max-width:980px;margin:18px auto;padding:18px}
        .card{border-radius:12px;padding:18px;background:#fafafa;box-shadow:0 6px 18px rgba(0,0,0,0.05)}
        input[type=file]{display:block;margin-top:8px}
        button{background:#0b76ef;color:white;border:none;padding:10px 14px;border-radius:8px;cursor:pointer}
        .messages{margin:12px 0}
        .messages .error{color:#a94442}
        .messages .success{color:#3c763d}
        table{width:100%;border-collapse:collapse;margin-top:12px}
        th,td{padding:8px;border-bottom:1px solid #eee;text-align:left}
        .log{max-height:260px;overflow:auto;background:white;padding:8px;border-radius:8px}
        label{display:block;margin-top:8px}
        select, input[type=text]{margin-top:6px;padding:8px;border-radius:6px;border:1px solid #ddd;width:100%;box-sizing:border-box}
    </style>
</head>
<body>
    <h1>CSV → <em>d_jobs</em> uploader</h1>
    <div class="card">
        <p>Choose a CSV file whose headers match the expected columns (a subset is fine). Missing columns will be set to <strong>NULL</strong> and empty CSV values will be stored as <strong>NULL</strong>. The script will update/insert into <code>jobs</code> as described. <strong>st_photo, bd_photo, coord will always be NULL</strong>.</p>
        <form method="post" enctype="multipart/form-data">
            <label>Override Signature</label>
            <select name="override_signature">
                <option value="">-- Do Not Override --</option>
                <option>Rhedrick</option>
            </select>

            <label>Override TAT</label>
            <input type="text" name="override_tat" placeholder="Leave blank to use CSV value">

            <label>Override Status</label>
            <select name="override_status">
                <option value="">-- Do Not Override --</option>
                <option value="Pending">Pending</option>
                <option value="Approved">Approved</option>
                <option value="Rejected">Rejected</option>
            </select>

            <label>Override Client</label>
            <input type="text" name="override_client" placeholder="Leave blank to use CSV value">

            <label>Select CSV file</label>
            <input type="file" name="csvfile" accept=".csv,text/csv" required>
            <div style="margin-top:12px">
                <button type="submit">Upload & Import</button>
            </div>
        </form>

        <div class="messages">
            <?php foreach ($messages as $m): ?>
                <div class="<?php echo $m['type']; ?>"><?php echo htmlspecialchars($m['text']); ?></div>
            <?php endforeach; ?>
        </div>

        <?php if (!empty($results)): ?>
            <h3>Per-row results</h3>
            <div class="log">
                <table>
                    <thead><tr><th>#</th><th>trxnref</th><th>result</th><th>error</th></tr></thead>
                    <tbody>
                        <?php foreach ($results as $r): ?>
                            <tr>
                                <td><?php echo htmlspecialchars($r['row']); ?></td>
                                <td><?php echo htmlspecialchars($r['trxnref']); ?></td>
                                <td><?php echo htmlspecialchars($r['status']); ?></td>
                                <td><?php echo isset($r['error'])?htmlspecialchars($r['error']):''; ?></td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        <?php endif; ?>

        <h4 style="margin-top:18px">Notes & Assumptions</h4>
        <ul>
            <li>Missing CSV columns are stored as <strong>NULL</strong>.</li>
            <li>Empty CSV values are stored as <strong>NULL</strong>.</li>
            <li><code>st_photo</code>, <code>bd_photo</code>, <code>coord</code> are always stored as <strong>NULL</strong>.</li>
            <li>When a corresponding <code>jobs</code> row exists for a <code>trxnref</code>, the script sets <code>jobs.status</code> = 'Done' and pulls <code>agent</code> and <code>post_by</code> into the inserted <code>d_jobs</code> row when CSV doesn't provide them.</li>
            <li>If <code>jobs</code> row doesn't exist, a fallback row is inserted into <code>jobs</code> using available CSV data with <code>status</code>='Done'. <strong>jobs.landmark</strong> is forced to <code>'N/A'</code> (kept from earlier behavior).</li>
            <li>post_date is computed from visitation_date & visitation_time when available; otherwise today's date + a random time between 9am and now (capped at 6pm).</li>
            <li>Overrides (Signature, TAT, Status, Client) — if selected — will overwrite CSV values and will also apply to the inserted rows.</li>
        </ul>

    </div>
</body>
</html>
