<?php
session_start();

/**
 * upload_analyze.php
 * Upload CSV with headers: CAV ID, Account Name, Address, State
 * Analyze and classify as per user logic
 * No external libraries required
 */

$randomDescriptions = [
"White Bungalow","Cream Storey Building","Grey Bungalow","Blue Bungalow","Brown Storey Building",
"Cream Multiple Floors","Unpainted Bungalow","Painted cream Block of flat","Brown Bungalow","Cream Bungalow",
"Milk Bungalow","Cream Storey Building","White Bungalow","Faded Storey Building","Yellow And Brown Storey Building",
"Green Bungalow","Wine/ not painted Storey Building","Milk Bungalow","Cream Block of flat","Yellow Storey Building",
"Grey Storey Building","Faded Block of flat","Faded Bungalow","Grey Storey Building","Grey Bungalow",
"Butter Storey Building","Brown Bungalow","Faded Block of flat","Cream/green Duplex","Cream/brown Storey Building",
"Red Bungalow","Brown/yellow Bungalow","Cream color Duplex","Yellow Bungalow","Not painted Bungalow",
"Butter Bungalow","Brown and cream Storey Building","Brown and butter Bungalow","Blue Bungalow","Brown Storey Building",
"Offwhite Storey Building","Milk Block of flat","Not painted Storey Building","Green Storey Building","Wine /cream Storey Building",
"Cream and brown Storey Building","Brown bungalow Bungalow","Cream color Storey Building","Orange Storey Building",
"Yellow /green Bungalow","Cream Bungalow","White Duplex","Wine Storey Building","Cream Block of flat",
"Brown and cream Bungalow","Milk Block of flat","Brown/yellow Bungalow","Brown Multiple Floors","Cream n brown Storey Building",
"White and blue Bungalow","Cream color Bungalow","Cream and ash Storey Building","Brown/green Storey Building",
"Green/Brown Storey Building","Brown and cream Bungalow","Faded paint Storey Building","White/Grey/Brown Storey Building",
"White Storey Building","Sandy Brown Storey Building","Cream and brown Bungalow","Faded Bungalow","Painted cream Bungalow",
"Wine Storey Building","Light Gray Bungalow","Unpainted Bungalow","Unpainted Storey Building","Grey Bungalow","Brown Bungalow",
"Grey/cream Bungalow","Red Bungalow","Yellow Bungalow","Green Bungalow","Not painted Storey Building","Not painted Bungalow","Cream color Storey Building",
"Brown Storey Building","Yellow and neon green Storey Building","Yellow And Brown Storey Building","Light Yellow Storey Building","Cream and oxblood Storey Building","Cream Storey Building",
"Peach Storey Building","Blue Storey Building","Faded paint Bungalow","Wine Storey Building","Wine Bungalow","Cream Bungalow","Peach and Gray Storey Building","pink and yellow Storey Building",
"Green and oxblood Bungalow","Yellow and red gate Storey Building","Cream and red gate Storey Building","Sandy Brown Storey Building","White Storey Building","Yellow Storey Building",
"Light Gray and yellow Storey Building","Yellow and black gate Bungalow","Brown/yellow Bungalow","Cream/wine Bungalow","Cream color Bungalow","Grey Storey Building",
];

$businessKeywords = [
    'ltd','limited','co\.','company','plc','bank','school','ministry','hotel','hospital','market','store','stores',
    'restaurant','clinic','estate','llc','ng','association','club','embassy','council','bureau','mall','enterprise',
    'ventures','inc','industries','services','consult','consulting','factory','garage','project','projects',
    'limited liability','agency','union','office','offices','investment','foundation','federal','state','church',
    'cathedral','mosque','cooperative','cooperatives','coop','cms','society','commission','ngo','partnership'
];

$traceKeywords = ['street','st.','road','rd.','lane','ln.','avenue','ave.','crescent','close','way','boulevard','drive','dr.','court','junction','highway','estate'];

function is_business_name(string $name, array $keywords): bool {
    $n = trim(strtolower($name));
    if ($n === '') return true;
    foreach ($keywords as $kw) {
        if (preg_match("/\b{$kw}\b/i", $n)) return true;
    }
    if (preg_match('/[0-9&@#]/', $n)) return true;
    $words = preg_split('/\s+/', trim($name));
    $validWords = array();
    foreach ($words as $w) {
        if (preg_match('/^[A-Za-z\.\'-]+$/', $w)) $validWords[] = $w;
    }
    $count = count($validWords);
    if ($count >= 2 && $count <= 4) return false;
    return true;
}

function is_traceable_address(string $address, array $keywords): bool {
    $a = strtolower(trim($address));
    if ($a === '' || strlen($a) < 6) return false;
    if (!preg_match('/\d+/', $a)) return false;
    foreach ($keywords as $k) {
        if (str_contains($a, $k)) return true;
    }
    return false;
}

function random_description(array $list) {
    return $list[array_rand($list)];
}

$results = [];
$msg = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_FILES['csvfile'])) {
    $file = $_FILES['csvfile'];
    if ($file['error'] !== UPLOAD_ERR_OK) {
        $msg = "Error uploading file.";
    } else {
        $tmp = $file['tmp_name'];
        if (($handle = fopen($tmp, 'r')) !== false) {
            $header = fgetcsv($handle);
            $cols = array_map('strtolower', array_map('trim', $header));
            $map = array_flip($cols);

            $required = ['cav id','account name','address','state'];
            foreach ($required as $r) {
                if (!isset($map[$r])) {
                    $msg = "Missing required column: $r";
                    fclose($handle);
                    break;
                }
            }

            while (($row = fgetcsv($handle)) !== false) {
                if (count(array_filter($row)) == 0) continue;
                $cav = trim($row[$map['cav id']] ?? '');
                $acct = trim($row[$map['account name']] ?? '');
                $addr = trim($row[$map['address']] ?? '');
                $state = trim($row[$map['state']] ?? '');

                $isBiz = is_business_name($acct, $businessKeywords);
                $isTrace = is_traceable_address($addr, $traceKeywords);

                if (!$isTrace) {
                    // Could not locate address
                    $category = 'Could not locate address';
                    $addrDesc = $addr;
                    $comment = 'inspected';
                    $status = 'vague';
                    $acctOut = $acct;
                } else {
                    if ($isBiz) {
                        // Customer not known (retain original Account Name)
                        $category = 'Customer not known';
                        $addrDesc = random_description($randomDescriptions);
                        $comment = 'verified';
                        $status = 'unsuccessful';
                        $acctOut = $acct; // ✅ Retain original name here
                    } else {
                        // Address exists and customer is known
                        $category = 'Address exists and customer is known';
                        $addrDesc = random_description($randomDescriptions);
                        $comment = 'verified';
                        $status = 'successful';
                        $acctOut = $acct;
                    }
                }

                $results[] = [
                    'CAV ID' => $cav,
                    'Account Name' => $acctOut,
                    'Category' => $category,
                    'Address_Description' => $addrDesc,
                    'Comment' => $comment,
                    'Status' => $status,
                    'Address' => $addr,
                    'State' => $state
                ];
            }
            fclose($handle);
            $_SESSION['results'] = $results;
            $msg = count($results)." records processed successfully.";
        }
    }
}


if (isset($_GET['export']) && $_GET['export'] === 'csv') {
    $data = $_SESSION['results'] ?? [];
    if (empty($data)) {
        echo "No data to export.";
        exit;
    }
    header('Content-Type: text/csv');
    header('Content-Disposition: attachment; filename="cav_analysis_'.date('Ymd_His').'.csv"');
    $out = fopen('php://output', 'w');
    fputcsv($out, array_keys($data[0]));
    foreach ($data as $r) fputcsv($out, $r);
    fclose($out);
    exit;
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
<meta charset="UTF-8">
<title>CAV UBA Analyzer</title>
<meta name="viewport" content="width=device-width, initial-scale=1">
<link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/css/bootstrap.min.css" rel="stylesheet">
<link href="https://cdn.datatables.net/1.13.6/css/jquery.dataTables.min.css" rel="stylesheet">
<style>
body{background:#f8fafc;padding:20px;}
.card{box-shadow:0 4px 10px rgba(0,0,0,0.05);}
.badge-successful{background:#198754}
.badge-unsuccessful{background:#dc3545}
.badge-vague{background:#ffc107;color:#000}
</style>
</head>
<body>
<div class="container">
  <div class="card p-4 mb-4">
    <h3>CAV UBA File Analyzer</h3>
    <form method="post" enctype="multipart/form-data" class="mt-3 row g-2">
      <div class="col-md-8">
        <input type="file" name="csvfile" class="form-control" accept=".csv" required>
      </div>
      <div class="col-md-4">
        <button class="btn btn-primary w-100" type="submit">Upload & Analyze</button>
      </div>
    </form>
    <?php if($msg): ?>
      <div class="alert alert-info mt-3"><?=htmlspecialchars($msg)?></div>
    <?php endif; ?>
  </div>

  <?php if(!empty($_SESSION['results'])): ?>
  <div class="card p-3">
    <div class="d-flex justify-content-between align-items-center mb-2">
      <h5 class="mb-0">Results</h5>
      <a href="?export=csv" class="btn btn-success">Export CSV</a>
    </div>
    <div class="table-responsive">
      <table id="resTable" class="table table-striped display">
        <thead>
          <tr>
            <th>CAV ID</th>
            <th>Account Name</th>
            <th>Category</th>
            <th>Address_Description</th>
            <th>Comment</th>
            <th>Status</th>
            <th>Address</th>
            <th>State</th>
          </tr>
        </thead>
        <tbody>
          <?php foreach($_SESSION['results'] as $r): 
            $cls = $r['Status']=='successful'?'badge-successful':($r['Status']=='unsuccessful'?'badge-unsuccessful':'badge-vague');
          ?>
          <tr>
            <td><?=htmlspecialchars($r['CAV ID'])?></td>
            <td><?=htmlspecialchars($r['Account Name'])?></td>
            <td><?=htmlspecialchars($r['Category'])?></td>
            <td><?=htmlspecialchars($r['Address_Description'])?></td>
            <td><?=htmlspecialchars($r['Comment'])?></td>
            <td><span class="badge <?=$cls?>"><?=htmlspecialchars($r['Status'])?></span></td>
            <td><?=htmlspecialchars($r['Address'])?></td>
            <td><?=htmlspecialchars($r['State'])?></td>
          </tr>
          <?php endforeach; ?>
        </tbody>
      </table>
    </div>
  </div>
  <?php endif; ?>
</div>

<script src="https://code.jquery.com/jquery-3.7.1.min.js"></script>
<script src="https://cdn.datatables.net/1.13.6/js/jquery.dataTables.min.js"></script>
<script>
$(function(){
  $('#resTable').DataTable({pageLength:25});
});
</script>
</body>
</html>
